<?php
/**
 * Gateway de Pagamento PIX para WooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Classe do Berapay
 */
class WC_PIX_Gateway extends WC_Payment_Gateway {
    /**
     * Chave da API configurada no gateway
     * @var string
     */
    private $api_key = '';

    /**
     * URL da API configurada no gateway
     * @var string
     */
    private $api_url = '';

    /**
     * Modo sandbox (yes/no)
     * @var string
     */
    private $sandbox = 'yes';

    /**
     * Modo debug (yes/no)
     * @var string
     */
    private $debug = 'no';
    
    /**
     * Construtor
     */
    public function __construct() {
        $this->id = 'pix_gateway';
        $this->icon = '';
        $this->has_fields = true;
        $this->method_title = 'Berapay';
        $this->method_description = __('Gateway de pagamento PIX e cartão de crédito.', 'woocommerce-pix-gateway');
        
        // Configurações
        $this->init_form_fields();
        $this->init_settings();
        
        // Propriedades
        $this->title = 'Berapay';
        $this->description = '';
        
        // Carregar configurações diretamente das opções do gateway (salvas pelo WooCommerce)
        $this->api_key = (string) $this->get_option('api_key', '');
        $this->api_url = 'http://api.getpaybr.com/v1/gateway/processa_pagamento_api.php'; // URL fixa do Berapay
        $this->sandbox = (string) $this->get_option('sandbox', 'yes');
        $this->debug = (string) $this->get_option('debug', 'no');
        
        // Hooks
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action('woocommerce_api_wc_pix_gateway', array($this, 'webhook_handler'));
        add_action('woocommerce_receipt_' . $this->id, array($this, 'receipt_page'));
        add_action('wp_ajax_wc_pix_gateway_check_status', array($this, 'ajax_check_status'));
        add_action('wp_ajax_nopriv_wc_pix_gateway_check_status', array($this, 'ajax_check_status'));
    }

    /**
     * Validar campos do checkout
     */
    public function validate_fields() {
        $payment_type = isset($_POST['wc_pix_payment_type']) ? sanitize_text_field(wp_unslash($_POST['wc_pix_payment_type'])) : 'PIX';
        if (strtoupper($payment_type) !== 'CARTAO') {
            return true;
        }
        $holder = isset($_POST['wc_pix_card_holder']) ? trim(sanitize_text_field(wp_unslash($_POST['wc_pix_card_holder']))) : '';
        $number = isset($_POST['wc_pix_card_number']) ? preg_replace('/[^0-9]/', '', wp_unslash($_POST['wc_pix_card_number'])) : '';
        $expiry = isset($_POST['wc_pix_card_expiry']) ? trim(sanitize_text_field(wp_unslash($_POST['wc_pix_card_expiry']))) : '';
        $cvv    = isset($_POST['wc_pix_card_cvv']) ? preg_replace('/[^0-9]/', '', wp_unslash($_POST['wc_pix_card_cvv'])) : '';
        
        if ($holder === '' || strlen($number) < 13 || strlen($number) > 19) {
            wc_add_notice(__('Dados do cartão inválidos. Verifique o número e o nome.', 'woocommerce-pix-gateway'), 'error');
            return false;
        }
        if (!preg_match('/^(0[1-9]|1[0-2])\/[0-9]{2}$/', $expiry)) {
            wc_add_notice(__('Validade do cartão inválida (use MM/AA).', 'woocommerce-pix-gateway'), 'error');
            return false;
        }
        if (strlen($cvv) < 3 || strlen($cvv) > 4) {
            wc_add_notice(__('CVV inválido.', 'woocommerce-pix-gateway'), 'error');
            return false;
        }
        return true;
    }

    /**
     * Campos exibidos no checkout
     */
  
    public function payment_fields() {
        echo '<p class="wc-pix-description">' . esc_html__('Escolha a forma de pagamento e, se for cartão, preencha os dados.', 'woocommerce-pix-gateway') . '</p>';
        ?>
        <fieldset class="wc-pix-gateway-fields" style="background:#F8F9FA;border:2px solid #E9ECEF;border-radius:16px;padding:24px;box-shadow:0 4px 8px rgba(0, 0, 0, 0.1);margin-bottom:16px;">
            <div class="wc-pix-options" style="display:grid;grid-template-columns:1fr;gap:16px;margin-bottom:20px;">
                <label class="wc-pix-option" style="background:#F8F9FA;border:2px solid #6C757D;border-radius:12px;padding:16px;display:flex;align-items:center;gap:12px;cursor:pointer;transition:all 0.3s ease;">
                    <input style="margin:0;width:20px;height:20px;accent-color:#6C757D;" type="radio" name="wc_pix_payment_type" value="PIX" checked> <span style="font-weight:600;color:#495057;font-size:16px;">&nbsp;<?php echo esc_html__('Pagar com PIX', 'woocommerce-pix-gateway'); ?></span>
                </label>
                <label class="wc-pix-option" style="background:#ffffff;border:2px solid #DEE2E6;border-radius:12px;padding:16px;display:flex;align-items:center;gap:12px;cursor:pointer;transition:all 0.3s ease;">
                    <input style="margin:0;width:20px;height:20px;accent-color:#6C757D;" type="radio" name="wc_pix_payment_type" value="CARTAO"> <span style="font-weight:600;color:#495057;font-size:16px;">&nbsp;<?php echo esc_html__('Pagar com Cartão de Crédito', 'woocommerce-pix-gateway'); ?></span>
                </label>
            </div>
            <div id="wc-pix-card-fields" class="wc-pix-card" style="display:none;background:#F8F9FA;border:2px solid #E9ECEF;border-radius:12px;padding:20px;box-shadow:0 2px 4px rgba(0, 0, 0, 0.1);margin-top:16px;">
                <div class="wc-pix-row" style="margin-bottom:20px;">
                    <label for="wc_pix_card_number" style="display:block;font-size:12px;color:#616161;margin-bottom:8px;font-weight:600;text-transform:uppercase;letter-spacing:0.5px;">&nbsp;<?php echo esc_html__('Número do cartão', 'woocommerce-pix-gateway'); ?></label>
                    <input type="text" id="wc_pix_card_number" name="wc_pix_card_number" class="wc-pix-input" inputmode="numeric" autocomplete="cc-number" placeholder="0000 0000 0000 0000" style="width:100%;padding:16px 20px;border:2px solid #E0E0E0;border-radius:12px;background:#fff;font-size:16px;line-height:1.5;box-sizing:border-box;transition:all 0.3s ease;" />
                </div>
                <div class="wc-pix-row" style="margin-bottom:20px;">
                    <label for="wc_pix_card_holder" style="display:block;font-size:12px;color:#616161;margin-bottom:8px;font-weight:600;text-transform:uppercase;letter-spacing:0.5px;">&nbsp;<?php echo esc_html__('Nome impresso no cartão', 'woocommerce-pix-gateway'); ?></label>
                    <input type="text" id="wc_pix_card_holder" name="wc_pix_card_holder" class="wc-pix-input" autocomplete="cc-name" placeholder="<?php echo esc_attr__('Nome completo', 'woocommerce-pix-gateway'); ?>" style="width:100%;padding:16px 20px;border:2px solid #E0E0E0;border-radius:12px;background:#fff;font-size:16px;line-height:1.5;box-sizing:border-box;transition:all 0.3s ease;" />
                </div>
                <div class="wc-pix-row wc-pix-row-inline" style="display:grid;grid-template-columns:1fr 1fr;gap:16px;">
                    <div class="wc-pix-col" style="display:flex;flex-direction:column;">
                        <label for="wc_pix_card_expiry" style="display:block;font-size:12px;color:#616161;margin-bottom:8px;font-weight:600;text-transform:uppercase;letter-spacing:0.5px;">&nbsp;<?php echo esc_html__('Validade (MM/AA)', 'woocommerce-pix-gateway'); ?></label>
                        <input type="text" id="wc_pix_card_expiry" name="wc_pix_card_expiry" class="wc-pix-input" inputmode="numeric" autocomplete="cc-exp" placeholder="MM/AA" style="width:100%;padding:16px 20px;border:2px solid #E0E0E0;border-radius:12px;background:#fff;font-size:16px;line-height:1.5;box-sizing:border-box;transition:all 0.3s ease;" />
                    </div>
                    <div class="wc-pix-col" style="display:flex;flex-direction:column;">
                        <label for="wc_pix_card_cvv" style="display:block;font-size:12px;color:#616161;margin-bottom:8px;font-weight:600;text-transform:uppercase;letter-spacing:0.5px;">&nbsp;<?php echo esc_html__('CVV', 'woocommerce-pix-gateway'); ?></label>
                        <input type="password" id="wc_pix_card_cvv" name="wc_pix_card_cvv" class="wc-pix-input" inputmode="numeric" autocomplete="cc-csc" placeholder="CVV" style="width:100%;padding:16px 20px;border:2px solid #E0E0E0;border-radius:12px;background:#fff;font-size:16px;line-height:1.5;box-sizing:border-box;transition:all 0.3s ease;" />
                    </div>
                </div>
            </div>
        </fieldset>
        <script>
        (function(){
            function styleOptions(){
                try {
                    var labels = document.querySelectorAll('.wc-pix-option');
                    labels.forEach(function(lbl){
                        lbl.style.borderColor = '#DEE2E6';
                        lbl.style.background = '#FFFFFF';
                        lbl.style.boxShadow = 'none';
                        lbl.style.transform = 'translateY(0)';
                    });
                    var sel = document.querySelector('input[name="wc_pix_payment_type"]:checked');
                    if (sel && sel.closest) {
                        var active = sel.closest('label');
                        if (active) {
                            active.style.borderColor = '#6C757D';
                            active.style.background = '#F8F9FA';
                            active.style.boxShadow = '0 2px 4px rgba(0, 0, 0, 0.1)';
                        }
                    }
                    var box = document.getElementById('wc-pix-card-fields');
                    if (box) {
                        box.style.borderColor = (sel && sel.value === 'CARTAO') ? '#6C757D' : '#E9ECEF';
                        box.style.background = (sel && sel.value === 'CARTAO') ? '#F8F9FA' : '#F8F9FA';
                    }
                } catch(e) {}
            }

            function toggleCard(){
                var sel = document.querySelector('input[name="wc_pix_payment_type"]:checked');
                var isCard = sel && sel.value === 'CARTAO';
                var box = document.getElementById('wc-pix-card-fields');
                if (box) {
                    box.style.display = isCard ? 'block' : 'none';
                    // Adicionar animação suave
                    if (isCard) {
                        box.style.opacity = '0';
                        box.style.transform = 'translateY(-10px)';
                        setTimeout(function() {
                            box.style.transition = 'all 0.3s ease';
                            box.style.opacity = '1';
                            box.style.transform = 'translateY(0)';
                        }, 10);
                    }
                }
                styleOptions();
            }
            
            document.addEventListener('change', function(e){
                if (e.target && e.target.name === 'wc_pix_payment_type') { 
                    toggleCard(); 
                }
            });
            
            // Inicializar
            toggleCard();

            // Máscaras melhoradas com validação
            var number = document.getElementById('wc_pix_card_number');
            var expiry = document.getElementById('wc_pix_card_expiry');
            var cvv = document.getElementById('wc_pix_card_cvv');
            
            if (number) {
                number.addEventListener('input', function(){
                    var v = this.value.replace(/[^0-9]/g,'').slice(0,19);
                    var out = v.match(/.{1,4}/g);
                    this.value = out ? out.join(' ') : v;
                    
                    // Validação visual
                    if (v.length >= 13 && v.length <= 19) {
                        this.style.borderColor = '#4CAF50';
                    } else if (v.length > 0) {
                        this.style.borderColor = '#FF9800';
                    } else {
                        this.style.borderColor = '#E0E0E0';
                    }
                });
                
                number.addEventListener('blur', function(){
                    var v = this.value.replace(/[^0-9]/g,'');
                    if (v.length > 0 && (v.length < 13 || v.length > 19)) {
                        this.style.borderColor = '#F44336';
                    }
                });
            }
            
            if (expiry) {
                expiry.addEventListener('input', function(){
                    var v = this.value.replace(/[^0-9]/g,'').slice(0,4);
                    if (v.length >= 2) {
                        v = v.slice(0,2) + '/' + v.slice(2);
                    }
                    this.value = v;
                    
                    // Validação básica
                    if (v.length === 5) {
                        var month = parseInt(v.slice(0,2));
                        var year = parseInt(v.slice(3,5));
                        var currentYear = new Date().getFullYear() % 100;
                        
                        if (month >= 1 && month <= 12 && year >= currentYear) {
                            this.style.borderColor = '#4CAF50';
                        } else {
                            this.style.borderColor = '#F44336';
                        }
                    } else if (v.length > 0) {
                        this.style.borderColor = '#FF9800';
                    } else {
                        this.style.borderColor = '#E0E0E0';
                    }
                });
            }
            
            if (cvv) {
                cvv.addEventListener('input', function(){
                    this.value = this.value.replace(/[^0-9]/g,'').slice(0,4);
                    
                    // Validação visual
                    if (this.value.length >= 3) {
                        this.style.borderColor = '#4CAF50';
                    } else if (this.value.length > 0) {
                        this.style.borderColor = '#FF9800';
                    } else {
                        this.style.borderColor = '#E0E0E0';
                    }
                });
            }
            
            // Melhorar responsividade dos campos
            function adjustFieldSizes() {
                var container = document.querySelector('.wc-pix-gateway-fields');
                if (container) {
                    var width = window.innerWidth;
                    var fields = container.querySelectorAll('.wc-pix-input');
                    
                    fields.forEach(function(field) {
                        if (width <= 480) {
                            field.style.fontSize = '16px'; // Previne zoom no iOS
                        } else if (width <= 768) {
                            field.style.fontSize = '14px';
                        } else {
                            field.style.fontSize = '16px';
                        }
                    });
                }
            }
            
            // Ajustar tamanhos na inicialização e redimensionamento
            adjustFieldSizes();
            window.addEventListener('resize', adjustFieldSizes);
            
        })();
        </script>
        <?php
    }
    
    /**
     * Campos de configuração do admin
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Habilitar/Desabilitar', 'woocommerce-pix-gateway'),
                'type' => 'checkbox',
                'label' => __('Habilitar Berapay', 'woocommerce-pix-gateway'),
                'default' => 'yes'
            ),
            'api_key' => array(
                'title' => __('API Key', 'woocommerce-pix-gateway'),
                'type' => 'text',
                'description' => __('Sua chave de API única fornecida pelo Berapay.', 'woocommerce-pix-gateway'),
                'default' => '',
                'desc_tip' => true,
            ),
            'sandbox' => array(
                'title' => __('Modo Sandbox', 'woocommerce-pix-gateway'),
                'type' => 'checkbox',
                'label' => __('Habilitar modo de teste', 'woocommerce-pix-gateway'),
                'default' => 'yes',
                'description' => __('Use para testar pagamentos sem processar valores reais.', 'woocommerce-pix-gateway'),
            ),
            'debug' => array(
                'title' => __('Modo Debug', 'woocommerce-pix-gateway'),
                'type' => 'checkbox',
                'label' => __('Habilitar logs de debug', 'woocommerce-pix-gateway'),
                'default' => 'no',
                'description' => __('Registra informações detalhadas para debugging.', 'woocommerce-pix-gateway'),
            ),
        );
    }
    
    /**
     * Processar pagamento
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            return array(
                'result' => 'failure',
                'redirect' => ''
            );
        }
        
        try {
            // Tipo de pagamento escolhido
            $payment_type = isset($_POST['wc_pix_payment_type']) ? sanitize_text_field(wp_unslash($_POST['wc_pix_payment_type'])) : 'PIX';
            $payment_type = strtoupper($payment_type) === 'CARTAO' ? 'CARTAO' : 'PIX';

            $card_data = array();
            if ($payment_type === 'CARTAO') {
                $card_data = array(
                    'cardHolder' => isset($_POST['wc_pix_card_holder']) ? sanitize_text_field(wp_unslash($_POST['wc_pix_card_holder'])) : '',
                    'cardNumber' => isset($_POST['wc_pix_card_number']) ? preg_replace('/[^0-9]/', '', wp_unslash($_POST['wc_pix_card_number'])) : '',
                    'cardExpiry' => isset($_POST['wc_pix_card_expiry']) ? sanitize_text_field(wp_unslash($_POST['wc_pix_card_expiry'])) : '',
                    'cardCvv' => isset($_POST['wc_pix_card_cvv']) ? preg_replace('/[^0-9]/', '', wp_unslash($_POST['wc_pix_card_cvv'])) : '',
                );
            }
            // Criar pagamento no gateway
            $payment_data = $this->create_payment($order, $payment_type, $card_data);
            
            if ($payment_data && isset($payment_data['status']) && $payment_data['status'] === 'success') {
                // Salvar dados do pagamento no pedido
                $order->update_meta_data('_pix_payment_id', $payment_data['idTransaction']);
                
                if (isset($payment_data['paymentCode'])) {
                    $order->update_meta_data('_pix_payment_code', $payment_data['paymentCode']);
                }
                
                if (isset($payment_data['qrCodeImage'])) {
                    $order->update_meta_data('_pix_qr_code_image', $payment_data['qrCodeImage']);
                }
                
                if (isset($payment_data['asaas'])) {
                    $order->update_meta_data('_pix_asaas_data', json_encode($payment_data['asaas']));
                }
                
                // Atualizar status do pedido conforme o tipo
                if ($payment_type === 'CARTAO') {
                    $asaas_status = isset($payment_data['asaas']['status']) ? strtoupper((string)$payment_data['asaas']['status']) : '';
                    if (in_array($asaas_status, array('CONFIRMED', 'RECEIVED', 'RECEIVED_IN_CASH', 'PAID_OUT', 'PAID'), true)) {
                        $order->payment_complete($payment_data['idTransaction']);
                        $order->add_order_note(__('Pagamento com cartão aprovado.', 'woocommerce-pix-gateway'));
                        $order->save();
                        
                        // Para pagamentos com cartão aprovados, redirecionar para página de pedido recebido
                        return array(
                            'result' => 'success',
                            'redirect' => $order->get_checkout_order_received_url()
                        );
                    } else {
                        $order->update_status('processing', __('Pagamento com cartão iniciado. Aguardando confirmação.', 'woocommerce-pix-gateway'));
                        $order->save();
                        
                        // Para pagamentos com cartão pendentes, redirecionar para página de pagamento
                        return array(
                            'result' => 'success',
                            'redirect' => $order->get_checkout_payment_url(true)
                        );
                    }
                } else {
                    $order->update_status('pending', __('Aguardando pagamento PIX.', 'woocommerce-pix-gateway'));
                    $order->save();
                    
                    // Para pagamentos PIX, redirecionar para página de pagamento
                    return array(
                        'result' => 'success',
                        'redirect' => $order->get_checkout_payment_url(true)
                    );
                }
            } else {
                // Construir mensagem de erro detalhada quando a API retornar erro
                $error_detail = '';
                if (is_array($payment_data)) {
                    if (isset($payment_data['message']) && $payment_data['message']) {
                        $error_detail = (string) $payment_data['message'];
                    }
                    if (isset($payment_data['asaas']['errors'][0]['description']) && $payment_data['asaas']['errors'][0]['description']) {
                        $error_detail = $error_detail ? ($error_detail . ' - ' . (string) $payment_data['asaas']['errors'][0]['description']) : (string) $payment_data['asaas']['errors'][0]['description'];
                    }
                }

                $base_msg = ($payment_type === 'CARTAO')
                    ? __('Erro ao criar pagamento no cartão.', 'woocommerce-pix-gateway')
                    : __('Erro ao criar pagamento PIX.', 'woocommerce-pix-gateway');

                $final_msg = trim($base_msg . ' ' . $error_detail);
                throw new Exception($final_msg);
            }
            
        } catch (Exception $e) {
            $this->log('Erro ao processar pagamento: ' . $e->getMessage());

            // Determinar o tipo escolhido para exibir a mensagem correta
            $chosen_type = isset($_POST['wc_pix_payment_type']) ? strtoupper(sanitize_text_field(wp_unslash($_POST['wc_pix_payment_type']))) : 'PIX';
            
            $detail = trim((string) $e->getMessage());
            
            // Se a mensagem já contém detalhes específicos do erro, mostrar apenas ela
            if ($detail && (stripos($detail, 'Erro ao criar cobrança') !== false || 
                           stripos($detail, 'billingType') !== false ||
                           stripos($detail, 'asaas') !== false ||
                           stripos($detail, 'estado de residência') !== false ||
                           stripos($detail, 'Informe o') !== false)) {
                wc_add_notice($detail, 'error');
            } else {
                // Caso contrário, mostrar mensagem genérica
                $user_base_msg = ($chosen_type === 'CARTAO')
                    ? __('Erro ao processar pagamento no cartão. Tente novamente.', 'woocommerce-pix-gateway')
                    : __('Erro ao processar pagamento PIX. Tente novamente.', 'woocommerce-pix-gateway');
                wc_add_notice($user_base_msg, 'error');
            }

            return array(
                'result' => 'failure',
                'redirect' => ''
            );
        }
    }
    
    /**
     * Criar pagamento no gateway
     */
    private function create_payment($order, $payment_type = 'PIX', $card_data = array()) {
        $api_url = $this->api_url;
        
        // Obter dados do cliente
        $billing_first_name = $order->get_billing_first_name();
        $billing_last_name = $order->get_billing_last_name();
        $billing_email = $order->get_billing_email();
        $billing_phone = $order->get_billing_phone();
        
        // Tentar obter CPF/CNPJ dos meta dados (compatível com diferentes plugins)
        $cpf_cnpj_raw = '';
        $possible_keys = array(
            '_billing_cpf', 'billing_cpf',
            '_billing_cnpj', 'billing_cnpj',
            '_cpf', 'cpf',
            '_cnpj', 'cnpj'
        );
        foreach ($possible_keys as $key) {
            $val = $order->get_meta($key);
            if (!empty($val)) {
                $cpf_cnpj_raw = (string) $val;
                break;
            }
        }
        
        // Formatar CPF/CNPJ com pontos e hífen (formato correto)
        $cpf_cnpj = preg_replace('/[^0-9]/', '', $cpf_cnpj_raw);
        if (strlen($cpf_cnpj) === 11) {
            // CPF: 000.000.000-00
            $cpf_cnpj = substr($cpf_cnpj, 0, 3) . '.' . substr($cpf_cnpj, 3, 3) . '.' . substr($cpf_cnpj, 6, 3) . '-' . substr($cpf_cnpj, 9, 2);
        } elseif (strlen($cpf_cnpj) === 14) {
            // CNPJ: 00.000.000/0000-00
            $cpf_cnpj = substr($cpf_cnpj, 0, 2) . '.' . substr($cpf_cnpj, 2, 3) . '.' . substr($cpf_cnpj, 5, 3) . '/' . substr($cpf_cnpj, 8, 4) . '-' . substr($cpf_cnpj, 12, 2);
        }
        
        // Formatar valor com vírgula (formato correto)
        $amount_formatted = number_format(floatval($order->get_total()), 2, ',', '.');
        
        // Converter tipo de pagamento para formato correto
        $tipo_pagamento = ($payment_type === 'CARTAO') ? 'CREDIT_CARD' : 'PIX';
        
        // Dados conforme o formato correto da API
        $payment_data = array(
            'api-key' => $this->api_key,
            'name' => trim($billing_first_name . ' ' . $billing_last_name),
            'cpf' => $cpf_cnpj,
            'amount' => $amount_formatted,
            'tipoPagamento' => $tipo_pagamento,
            'email' => $billing_email,
            'user_id' => 'gomes' // Campo obrigatório conforme exemplo correto
        );
        
        // Incluir dados do cartão quando aplicável (formato correto)
        if ($tipo_pagamento === 'CREDIT_CARD' && !empty($card_data)) {
            // Separar mês e ano da validade
            $expiry_parts = explode('/', $card_data['cardExpiry']);
            $expiry_month = isset($expiry_parts[0]) ? $expiry_parts[0] : '';
            $expiry_year = isset($expiry_parts[1]) ? $expiry_parts[1] : ''; // Manter apenas os 2 últimos dígitos
            
            $payment_data['creditCard'] = array(
                'number' => $card_data['cardNumber'],
                'expiryMonth' => $expiry_month,
                'expiryYear' => $expiry_year,
                'ccv' => $card_data['cardCvv']
            );
            
            // Formatar telefone para o padrão correto
            $formatted_phone = '';
            if (!empty($billing_phone)) {
                $phone_clean = preg_replace('/[^0-9]/', '', $billing_phone);
                if (strlen($phone_clean) >= 10) {
                    if (strlen($phone_clean) === 11) {
                        $formatted_phone = '(' . substr($phone_clean, 0, 2) . ') ' . substr($phone_clean, 2, 5) . '-' . substr($phone_clean, 7, 4);
                    } else {
                        $formatted_phone = '(' . substr($phone_clean, 0, 2) . ') ' . substr($phone_clean, 2, 4) . '-' . substr($phone_clean, 6, 4);
                    }
                } else {
                    $formatted_phone = $billing_phone;
                }
            }
            
            $payment_data['creditCardHolderInfo'] = array(
                'name' => $card_data['cardHolder'],
                'email' => $billing_email,
                'cpfCnpj' => $cpf_cnpj,
                'phone' => $formatted_phone,
                'address' => $order->get_billing_address_1(),
                'addressNumber' => $order->get_meta('_billing_number') ?: 'S/N',
                'city' => $order->get_billing_city(),
                'state' => $order->get_billing_state(),
                'postalCode' => preg_replace('/[^0-9]/', '', $order->get_billing_postcode())
            );
        }
        
        // Formatar telefone para o padrão correto (11) 99999-9999
        if (!empty($billing_phone)) {
            $phone_clean = preg_replace('/[^0-9]/', '', $billing_phone);
            if (strlen($phone_clean) >= 10) {
                // Se tem 11 dígitos (com DDD), formatar como (XX) 99999-9999
                if (strlen($phone_clean) === 11) {
                    $formatted_phone = '(' . substr($phone_clean, 0, 2) . ') ' . substr($phone_clean, 2, 5) . '-' . substr($phone_clean, 7, 4);
                } else {
                    // Se tem 10 dígitos, formatar como (XX) 9999-9999
                    $formatted_phone = '(' . substr($phone_clean, 0, 2) . ') ' . substr($phone_clean, 2, 4) . '-' . substr($phone_clean, 6, 4);
                }
                $payment_data['telefone'] = $formatted_phone;
            } else {
                $payment_data['telefone'] = $billing_phone;
            }
        }
        
        $this->log('Enviando requisição para: ' . $api_url);
        $this->log('Dados do pagamento: ' . print_r($payment_data, true));
        
        $response = wp_remote_post($api_url, array(
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'Accept' => 'application/json'
            ),
            'body' => json_encode($payment_data),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            $this->log('Erro na requisição: ' . $response->get_error_message());
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $status_code = wp_remote_retrieve_response_code($response);
        
        $this->log('Resposta do gateway - Status: ' . $status_code . ' - Body: ' . $body);
        
        if ($status_code === 200) {
            $data = json_decode($body, true);
            return $data;
        }
        
        return false;
    }
    
    /**
     * Consultar status do pagamento
     */
    private function check_payment_status($payment_id) {
        // Montar URL de consulta via GET (ex.: https://.../processa_pagamento_api.php?api-key=XXX&idTransaction=YYY)
        $base_url = preg_replace('/[?&]+$/', '', (string) $this->api_url);
        
     
        $api_url = add_query_arg(array(
            'api-key' => (string) $this->api_key,
            'idTransaction' => (string) $payment_id,
        ), $base_url);
        
        $this->log('Consultando status (GET): ' . $api_url);
        
        $response = wp_remote_get($api_url, array(
            'timeout' => 15,
            'headers' => array(
                'Accept' => 'application/json'
            )
        ));
        
        if (is_wp_error($response)) {
            $this->log('Erro na consulta de status: ' . $response->get_error_message());
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $status_code = wp_remote_retrieve_response_code($response);
        
        $this->log('Resposta da consulta - Status: ' . $status_code . ' - Body: ' . $body);
        
        if ($status_code === 200) {
            $data = json_decode($body, true);
            return $data;
        }
        
        return false;
    }
    
    /**
     * Página de recebimento
     */
    public function receipt_page($order_id) {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            echo '<p>' . __('Pedido não encontrado.', 'woocommerce-pix-gateway') . '</p>';
            return;
        }
        
        $payment_id = $order->get_meta('_pix_payment_id');
        $payment_code = $order->get_meta('_pix_payment_code');
        $qr_code_image = $order->get_meta('_pix_qr_code_image');
        
        if (!$payment_id) {
            echo '<p>' . __('Dados do pagamento não encontrados.', 'woocommerce-pix-gateway') . '</p>';
            return;
        }
        
        include WC_PIX_GATEWAY_PLUGIN_PATH . 'templates/payment-page.php';
    }
    
    /**
     * Handler do webhook
     */
    public function webhook_handler() {
        $this->log('Webhook recebido: ' . print_r($_POST, true));
        
        if (!isset($_POST['idTransaction']) || !isset($_POST['status'])) {
            $this->log('Webhook inválido - dados ausentes');
            status_header(400);
            exit;
        }
        
        $transaction_id = sanitize_text_field($_POST['idTransaction']);
        $status = sanitize_text_field($_POST['status']);
        
        // Buscar pedido pelo ID da transação
        $orders = wc_get_orders(array(
            'meta_key' => '_pix_payment_id',
            'meta_value' => $transaction_id,
            'limit' => 1
        ));
        
        if (empty($orders)) {
            $this->log('Pedido não encontrado para transação: ' . $transaction_id);
            status_header(404);
            exit;
        }
        
        $order = $orders[0];
        
        // Atualizar status do pedido conforme a documentação
        switch ($status) {
            case 'CONFIRMED':
                $order->payment_complete($transaction_id);
                $order->add_order_note(__('Pagamento PIX confirmado via webhook.', 'woocommerce-pix-gateway'));
                break;
                
            case 'OVERDUE':
                $order->update_status('failed', __('Pagamento PIX vencido.', 'woocommerce-pix-gateway'));
                break;
                
            case 'CANCELLED':
                $order->update_status('cancelled', __('Pagamento PIX cancelado.', 'woocommerce-pix-gateway'));
                break;
                
            case 'REFUNDED':
                $order->update_status('refunded', __('Pagamento PIX estornado.', 'woocommerce-pix-gateway'));
                break;
                
            default:
                $this->log('Status não reconhecido: ' . $status);
        }
        
        $this->log('Webhook processado com sucesso para pedido: ' . $order->get_id());
        
        status_header(200);
        echo 'OK';
        exit;
    }
    
    /**
     * AJAX para verificar status do pagamento
     */
    public function ajax_check_status() {
        // Validar nonce sem encerrar a execução para evitar respostas não-JSON no frontend
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'wc_pix_gateway_nonce')) {
            $this->log('AJAX check status: nonce inválido');
            // seguimos adiante para não quebrar a UX; admin-ajax tradicional mataria a resposta com -1
        }
        
        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        
        if (!$order_id) {
            wp_send_json_error(__('ID do pedido inválido.', 'woocommerce-pix-gateway'));
        }
        
        $order = wc_get_order($order_id);
        
        if (!$order) {
            wp_send_json_error(__('Pedido não encontrado.', 'woocommerce-pix-gateway'));
        }
        
        $payment_id = $order->get_meta('_pix_payment_id');
        
        if (!$payment_id) {
            wp_send_json_error(__('ID da transação não encontrado.', 'woocommerce-pix-gateway'));
        }
        
        // Montar URL de debug igual ao usado internamente
        $debug_base_url = preg_replace('/[?&]+$/', '', (string) $this->api_url);
        $debug_url = add_query_arg(array(
            'api-key' => (string) $this->api_key,
            'idTransaction' => (string) $payment_id,
        ), $debug_base_url);

        $status_data = $this->check_payment_status($payment_id);
        
        if ($status_data && isset($status_data['status']) && $status_data['status'] === 'success') {
            // Tentar obter o status prioritariamente do bloco asaas, senão do bloco pedido
            $remote_status = '';
            if (isset($status_data['asaas']['status'])) {
                $remote_status = (string) $status_data['asaas']['status'];
            } elseif (isset($status_data['pedido']['status'])) {
                $remote_status = (string) $status_data['pedido']['status'];
            }
            $remote_status_upper = strtoupper($remote_status ?: 'PENDING');
            
            // Normalizar status do gateway para o nosso domínio
            $status_confirmeds = array('CONFIRMED', 'RECEIVED', 'RECEIVED_IN_CASH', 'RECEIVED_IN_CASH_UNCONFIRMED', 'PAID_OUT', 'PAID');
            $status_pendings = array('PENDING', 'WAITING_FOR_APPROVAL', 'AWAITING_PAYMENT');
            $status_overdues  = array('OVERDUE', 'EXPIRED');
            $status_cancelled = array('CANCELLED', 'CANCELED');

            $normalized_status = 'PENDING';
            if (in_array($remote_status_upper, $status_confirmeds, true)) {
                $normalized_status = 'CONFIRMED';
            } elseif (in_array($remote_status_upper, $status_overdues, true)) {
                $normalized_status = 'OVERDUE';
            } elseif (in_array($remote_status_upper, $status_cancelled, true)) {
                $normalized_status = 'CANCELLED';
            } else {
                $normalized_status = 'PENDING';
            }
            
            $is_confirmed = ($normalized_status === 'CONFIRMED');
            $is_overdue = ($normalized_status === 'OVERDUE');
            $is_cancelled = ($normalized_status === 'CANCELLED');
            
            if ($is_confirmed && $order->get_status() === 'pending') {
                $order->payment_complete($payment_id);
                $order->add_order_note(__('Pagamento PIX confirmado via verificação automática.', 'woocommerce-pix-gateway'));
                
                wp_send_json_success(array(
                    'status' => 'CONFIRMED',
                    'redirect_url' => $order->get_checkout_order_received_url(),
                    'debug_url' => $debug_url,
                    'api_base_url' => $debug_base_url,
                ));
            }
            
            if ($is_overdue && !in_array($order->get_status(), array('failed', 'cancelled', 'refunded'), true)) {
                $order->update_status('failed', __('Pagamento PIX vencido.', 'woocommerce-pix-gateway'));
            }
            
            if ($is_cancelled && $order->get_status() !== 'cancelled') {
                $order->update_status('cancelled', __('Pagamento PIX cancelado.', 'woocommerce-pix-gateway'));
            }
            
            wp_send_json_success(array(
                'status' => $normalized_status,
                'debug_url' => $debug_url,
                'api_base_url' => $debug_base_url,
            ));
        } else {
            $this->log('AJAX check status: resposta inválida do gateway: ' . print_r($status_data, true));
            wp_send_json_error(array(
                'message' => __('Erro ao verificar status do pagamento.', 'woocommerce-pix-gateway'),
                'debug_url' => $debug_url,
                'api_base_url' => $debug_base_url,
            ));
        }
    }
    
    /**
     * Log de debug
     */
    private function log($message) {
        if ($this->debug === 'yes') {
            $logger = wc_get_logger();
            $logger->info($message, array('source' => 'pix_gateway'));
        }
    }
}
